import spaceport.Spaceport
import spaceport.bridge.Command
import spaceport.launchpad.Launchpad
import spaceport.computer.alerts.Alert
import spaceport.computer.alerts.results.Result
import spaceport.computer.alerts.results.HttpResult

/*
 *    _              /_
 *  _)/)(/(_(-/)()/-/
 *   /       /    v2
 *
 * This file serves as the entry point for this application using the Spaceport framework.
 * The Alert hooks contained are used in the spaceport lifecycle to enable the application to
 * be notified of certain events, and presented an opportunity to mutate the result.
 *
 * Developers:
 *
 * For more information on using the Spaceport Router, please see the documentation at:
 * https://www.spaceport.com.co/docs/router.html
 *
 * For more information on using the Spaceport Alert hooks, please see the documentation at:
 * https://www.spaceport.com.co/docs/alerts.html
 *
 */


class Router {


    // It's a good idea to define the name and version of your application
    // and have a central place to change and reference it.
    static final name    = "[NAME]"
    static final version = "1.0.0"


    // the 'on initialize' Alert is called once when the Spaceport starts, and subsequently when
    // the Spaceport PartsStore is reloaded. Likewise, 'on deinitialize' is called when the Spaceport
    // is stopped, and right before the Spaceport PartsStore is reloaded.
    @Alert('on initialize')
    static _init(Result r) {
        //
        // Add your initialization code here

        Command.println("Initializing $name v$version")

        //
        // Perhaps instantiate a database?

        // def databaseId = "[NAME]"
        // if (!Spaceport.main_memory_core.containsDatabase(databaseId)) {
        //    Spaceport.main_memory_core.createDatabase(databaseId)
        //    Command.debug("[YELLOW]Created database [REVERSE] $databaseId [RESET]")
        // }
    }


    // Use Alerts for Routing, as well, by using the on hit Alert. This is
    // called on any HTTP request (hit) for the root of the spaceport.
    @Alert('on / hit')
    static _root(HttpResult r) {
        // Set the result of the HTTP request to redirect to index.html
        r.setRedirectUrl('/index.html')
    }


    // Called when /index.html is hit. [index.html] may be replaced with any
    // other naming construct you require.
    @Alert('on /index.html hit')
    static _index(HttpResult r) {
        // Write a File to the client and Spaceport will automatically set the
        // content-type header based on the file extension.

        // You can write any file to the client, but we'll use a handy
        // method to get a file from the root of the spaceport and serve
        // up a very vanilla HTML file.
        r.writeToClient(Spaceport.getSpaceportFile('assets/index.html'))
    }


    // Serve up JSON content for an API/REST-like endpoint by sending a Map object
    // to the client, and specifying the type of HTTP hit. In this case, only a 'GET'
    // is handled. 'POST', 'PUT', 'DELETE', etc. can be handled in the same way.
    @Alert('on /api/version get')
    static _version(HttpResult r) {
        def json = [
                name: name,
                version: version,
                platform: "Spaceport v2"
        ]

        // Writing a Map to the client will automatically set the content-type
        // header to 'application/json', and the response body will be the
        // JSON representation of the Map.
        r.writeToClient(json)
    }



    //
    //   /        /      /
    //  /(////)(_/)/)(/(/
    //            /
    //
    // Launchpad provides powerful hypermedia content using spaceport templates.
    //
    // For more information on using the Launchpad, please see the documentation at:
    // https://www.spaceport.com.co/docs/launchpad.html


    // Use a launchpad to serve up the content using Spaceport Templates—
    // a combination of HTML, CSS, JavaScript and Server-Side Groovy paired with
    // other spaceport features to enable a dynamic and interactive development
    // experience, and to serve up enhanced hypermedia content to the client.

    static launchpad = new Launchpad()


    // HTTP hit alerts can match a regular expression to serve up a response.
    // In this case, we're accepting any path in the /x/ root.
    // The '~' prefix is used to denote the path uses regular expressions, and ()'s are used
    // to establish capture groups, which can be referenced using HttpResult.matches[index]

    @Alert('~on /\\*/(tutorials|tests|command)/(.*) hit')
    static _page(HttpResult r) {
        // Grab the 'page' requested from the first capture group
        def section = r.matches[0]
        def page    = r.matches[1]

        def sectionMap = [
                'tutorials': 'tutorial-pages',
                'tests': 'tests',
                'command': 'command-module'
        ]

        // Assemble the .ghtml files (spaceport-enhanced groovy template) specified
        // in the request URI. The path is relative to the 'parts' directory in the
        // launchpad's source path, in this case the module's directory.
        def payload = launchpad.assemble([ sectionMap[section] + '/' + page + '.ghtml'])

        // Launch the payload, committing the response to the client though the result.
        payload.launch(r, sectionMap[section] + '/_wrapper.ghtml')
    }


    // Provides some basic 'fallback' functionality for responses that
    // are not handled by the application including basic 404 and 50X
    // pages, as well as serving favicon.ico.
    @Alert('on page hit')
    static _hit(HttpResult r) {

        // Only handle non-200 responses
        if (r.context.response.status == 200) return

        // Serve favicon.ico
        if (r.context.target == '/favicon.ico') {
            // Redirect to favicon
            r.setRedirectUrl("/assets/favicon.svg")
            return
        }

        // No need to continue if the response is already committed
        if (r.context.response.isCommitted()) return

        // Show internal error(s)
        if (r.context.response.status >= 500) {
            def status = r.context.response.status
            // Clear the response that resulted in an error
            r.context.response.reset()
            r.setStatus(status)
            launchpad.assemble(['error-pages/50X.ghtml']).launch(r)
        }

        // ...and handle not found/bad request(s)
        else if (r.context.response.status >= 400) {
            // Keep the status
            def status = r.context.response.status
            // Reset the response to render the error page
            r.context.response.reset()
            r.setStatus(status)
            launchpad.assemble(['error-pages/40X.ghtml']).launch(r)
        }
    }

}