import spaceport.Spaceport
import spaceport.bridge.Command
import spaceport.launchpad.SpaceportTemplateEngine
import spaceport.computer.alerts.Alert
import spaceport.computer.alerts.results.Result
import spaceport.computer.memory.physical.CouchHandler
import spaceport.engineering.SourceStore

import java.nio.file.FileSystems
import java.nio.file.FileVisitResult
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.SimpleFileVisitor
import java.nio.file.WatchEvent
import java.nio.file.WatchKey
import java.nio.file.WatchService
import java.nio.file.attribute.BasicFileAttributes

import static java.nio.file.StandardWatchEventKinds.ENTRY_CREATE
import static java.nio.file.StandardWatchEventKinds.ENTRY_DELETE
import static java.nio.file.StandardWatchEventKinds.ENTRY_MODIFY
import static java.nio.file.StandardWatchEventKinds.OVERFLOW


/*
 *   _              /_
 * _)/)(/(_(-/)()/-/   COMMAND MONITOR
 *  /       /    v2
 *
 * This file provides a file monitor for active development and debugging. It can be turned off
 * in a production setting, but could also be a starting point for your own file watcher with
 * your very own special criteria.
 *
 * Developers:
 *
 * For more information on using the Command Monitor, please see the documentation at:
 * https://www.spaceport.com.co/docs/monitor.html
 *
 */



class Scanner {

    static boolean ENABLED = true

    @Alert('on initialize')
    static _initialize(Result r) {

        if (!ENABLED) return

        //
        // Watch the /modules/ directory for changes to .groovy files, then perform a soft reload
        //
        new Thread().start { // Start a new Thread to watch the directory, this is blocking otherwise

            def path = Paths.get(Spaceport.config.'spaceport root', Spaceport.config.'spaceport name', 'modules')
            def watcher = new Scanner(path, { Path p, def k ->
                        if (p.toString().endsWith('.groovy')) {
                            Command.debug('Reloading modules due to file change: ' + p.toString())
                            // Reload Modules by rescanning the PartsStore.
                            SourceStore.scan()
                            // Clear Document cache to take in outside database changes.
                            CouchHandler.cachedDocuments?.clear()
                            // Clear Template cache to allow for newer modules to be referenced.
                            SpaceportTemplateEngine.templateCache?.clear()
                        }
                    })

            watcher.watch()
            watched.add(watcher)
        }

        // Add your own watchers!

        // new Thread().start { // Start a new Thread to watch the directory, this is blocking otherwise
        //     def path = Paths.get( ... )
        //     def watcher = new Scanner(path, { Path path, def action ->
        //          ...
        //        })
        //     watcher.watch()
        //     watched.add(watcher)
        // }
    }


    @Alert('on deinitialize')
    static _deinitialize(Result r) {
        if (!watched.isEmpty()) {
            watched.each { Scanner watcher ->
                // The Thread will stop itself when the watcher is done
                watcher.stop()
            }
        }
    }


    //
    // Implement a basic Java File Watcher
    //


    static <T> WatchEvent<T> cast(WatchEvent<?> event) {
        return (WatchEvent<T>) event
    }

    static List<Scanner> watched = []  // Add multiple watchers


    /**
     * Creates a WatchService and calls action on events in the given directory.
     * @param dir The directory to watch.
     * @param action The closure to call on events.
     */
    Scanner(Path dir, Closure action) throws IOException {
        Command.debug('Monitoring ' + dir.toString() + ' for changes.')
        this.watcher = FileSystems.getDefault().newWatchService()
        this.keys = new HashMap<WatchKey,Path>()
        this.action = action
        scan(dir)
    }


    private WatchService watcher
    public Map<WatchKey,Path> keys = [:]
    private Closure action
    private def done = false


    private void scan(final Path start) throws IOException {
        Files.walkFileTree(start, new SimpleFileVisitor<Path>() {
            @Override
            FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                WatchKey key = dir.register(watcher, ENTRY_CREATE, ENTRY_DELETE, ENTRY_MODIFY)
                keys.put(key, dir)
                return FileVisitResult.CONTINUE
            }
        })
    }

    void watch() {
        while (!done) {
            WatchKey key
            try { key = watcher.take() } catch (InterruptedException x) {
                return
            }

            // Grab path
            Path dir = keys.get(key)
            if (dir == null) {
                continue
            }

            // Process events
            for (WatchEvent<?> event : key.pollEvents()) {
                WatchEvent.Kind kind = event.kind()
                if (kind == OVERFLOW) continue
                // Context for directory entry event is the file name of entry
                WatchEvent<Path> ev = cast(event)
                Path name = ev.context()
                Path child = dir.resolve(name)
                // Call action with path and event type
                action(child, kind)
            }

            // Check for key validity
            boolean valid = key.reset()
            if (!valid) {
                keys.remove(key)
                if (keys.isEmpty()) {
                    // No keys left to watch, stalled.
                    break
                }
            }
        }
    }


    void stop() {
        if (keys != null) {
            Iterator<WatchKey> it = keys.keySet().iterator()
            while (it.hasNext()) {
                WatchKey key = it.next()
                if (key != null) key.cancel()
            }
        }
        done = true
        if (watcher != null) {
            try {
                watcher.close()
            } catch (IOException e) {
                e.printStackTrace()
            }
        }
        Command.debug('Scanner stopped.')
    }


}