import spaceport.bridge.Command
import spaceport.computer.alerts.Alert
import spaceport.computer.alerts.Result
import spaceport.computer.alerts.results.HttpResult
import spaceport.personnel.Client

/**
 *
 *   _              /_
 * _)/)(/(_(-/)()/-/   MERCURY, a very basic scaffold
 *  /       /    v2
 *
 * This App file is a very basic 1-file starting point for a Spaceport application.
 * In larger code bases, it would be beneficial to separate the systems such as routing,
 * login process, etc.
 *
 * Mercury provides basic usage of the Alert system, HTTP Routing, and
 * basic client authentication. Use this scaffold to make a basic application
 * that doesn't require any users, but may have a 'back-end' that is available
 * to a 'spaceport-administrator' privileged user.
 *
 * For a more robust application, consider using the 'Pioneer' scaffold.
 *
 */
class App {


    // A hook that triggers when Spaceport starts
    @Alert('on initialized')
    static _init(Result r) {

        Command.debug('App initialized.')
    }


    // A simple HTTP route to the root
    @Alert('on / hit')
    static _root(HttpResult r) {

        if (r.client.isAuthenticated() && r.client.clientDocument.hasPermission('spaceport-administrator')) {
            // Must be authenticated, and have the 'spaceport-administrator' permission
            r.writeToClient("Hello ${r.client.userID}, from Spaceport.")
        } else {
            // Otherwise, just say hello.
            r.writeToClient('Hello World, from Spaceport.')
        }
    }


    // A simple login form, responding to GET on the /login route
    @Alert('on /login get')
    static _login(HttpResult r) {

        // Establish a bare-minimum login page
        def loginForm = /* language=HTML */ """
            <head><title>Login</title></head>
            <body>
                <h1>Login</h1>
                <form action='/login' method='POST'>
                    <label for="username">Username:</label><br>
                    <input type="text" id="username" name="username" required><br><br>
                    <label for="password">Password:</label><br>
                    <input type="password" id="password" name="password" required><br><br>
                    <input type="submit" value="Login">
                </form>
            </body> 
            """

        // Write it to the client
        r.writeToClient(loginForm)
    }


    // A simple login processor, responding to POST on the /login route
    @Alert('on /login post')
    static _processLogin(HttpResult r) {
        String username = r.context.data.'username'
        String password = r.context.data.'password'

        // Attempt to authenticate client
        Client.getAuthenticatedClient(username, password)?.tap {
            // Successful authentication

            // We now have an authenticated Client, let's attach
            // the UUID from the endpoint to the client's cookies
            // so that the client will be automatically
            // authenticated on future requests
            attachCookie(r.context.cookies.'spaceport-uuid')

            // Optional, but we can update the client's last login time
            // to the current time
            clientDocument.setState('last-login', System.currentTimeMillis(), [
                    'time' : System.currentTimeMillis(),
                    'ip'   : r.context.requestHeaders.'X-Forwarded-For',
                    'user-agent' : r.context.requestHeaders.'User-Agent'
                ])

            // Redirect the client to '/'
            r.setRedirectUrl('/')

            // Otherwise, the return value was null (unable to be authenticated)
            // so we'll write a failure message to the client
        } ?: r.writeToClient(/* language=HTML */ """
                <head><title>Login</title></head>
                <body>
                    <h1>Login Failed</h1>
                    <p>Incorrect user credentials. <a href='/login'>Try again?</a></p>
                </body> 
                """)

    }


    // A simple logout processor, responding to the /logout route
    @Alert('on /logout hit')
    static _logout(HttpResult r) {

        if (r.client.isAuthenticated()) {
            r.client.removeCookie(r.context.cookies.'spaceport-uuid')
            r.writeToClient(/* language=HTML */ """
                <head><title>Logout</title></head>
                <body>
                    <h1>Logout</h1>
                    <p>Goodbye ${ r.client.userID }. <a href='/login'>Login?</a> <a href="/">Go home?</a></p>
                </body> 
                """)

        } else {
            r.writeToClient(/* language=HTML */ """
                <head><title>Logout</title></head>
                <body>
                    <h1>Logout</h1>
                    <p>You are not logged in. <a href='/login'>Login?</a> <a href="/">Go home?</a></p>
                </body> 
                """)
        }

    }

}